% main script for linear power system state estimation

clear all;
close all;
clc;
addpath(genpath(pwd))

disp(' ');
disp('************************************************************');
disp('* This script solves the state estimation problem *');
disp('************************************************************');
disp(' ');

%%  Step 1: Load nodal admittance matrix and base values
%   The data need to be saved in the folder 'Data_SE'
%   in the same directory as this script.

disp('STEP 1: loading grid parameters ...');
disp(' ');

Y_LF = importdata('Data_SE/Y_14bus.mat');
A_b = importdata('Data_SE/Base_Power.mat');
V_b = importdata('Data_SE/Base_Voltage.mat');

n_system = size(Y_LF,1); % number of nodes (incl. slack)

disp(['The network has ' int2str(n_system) ' nodes (incl. slack).']);
disp(['The base power is ' num2str(A_b/1e6) ' MVA.']);
disp(['The base voltage is ' num2str(V_b/1e3) ' kV.']);
disp(' ');

Y_b = A_b/V_b^2; % base admittance

%%  Step 2: Construct the reduced nodal admittance matrix
%   Since the voltage of the slack node is fixed,
%   it does not need be estimated.

disp('STEP 2: constructing Y matrix ...');
disp(' ');

% Part A: equivalent circuit of the slack node

% Short-circuit parameters
S_sc = 300e6; % short-circuit power
r_sc = 0.1; % R/X ratio of the short-circuit impedance

X_sc =  V_b^2/(S_sc*sqrt(1+r_sc^2));
R_sc = r_sc*X_sc;
Z_sc = (R_sc + 1i*X_sc) * Y_b; % short-circuit impedance (absolute units)
Y_sc = 1/Z_sc;

% Part B: reduce the nodal admittance matrix

% Slack node
idx_LF_slack = 1;

% Neighbour of the slack node
idx_LF_neighbor = 2;

% Reduce the nodal admittance matrix
Y_SE = reduce_Y(Y_LF,Y_sc,idx_LF_slack,idx_LF_neighbor);

n_nodes = size(Y_SE,1); % Number of nodes (excl. slack)

disp(['The network has ' num2str(n_nodes) ' nodes (excl. slack)']);
disp(' ');

%%  Step 3: Load nodal power profiles
%   The data need to be saved in the folder 'Data_SE'
%   in the same directory as this script.
%   The data need to be given as matrices of size n_nodes x n_timesteps.

disp('STEP 3: loading PQ profiles ...')
disp(' ');

% Load profiles
P_abs = importdata('Data_SE/P_load.mat');
Q_abs = importdata('Data_SE/Q_load.mat');

% Generator profiles
P_inj = importdata('Data_SE/P_generation.mat');
Q_inj = importdata('Data_SE/Q_generation.mat');

n_timesteps = size(P_abs,2);

% Check number of nodes
if(~all([size(P_abs,1),size(Q_abs,1),size(P_inj,1),size(Q_inj,1)]==n_system))
    error('The nodal power profiles must have n_system rows.');
end

% Check number of timesteps
if(~all([size(P_abs,2),size(Q_abs,2),size(P_inj,2),size(Q_inj,2)]==n_timesteps))
    error('The nodal power profiles must have n_timesteps columns.');
end

% Manipulate the power profiles to create a suitable scenario
S_net = 0.5*complex(12*P_inj-P_abs,12*Q_inj-Q_abs)/A_b;

disp(['The profiles correspond to ' int2str(n_timesteps) ' timesteps.']);
disp(' ');

%%  Step 4: Configure state estimator

disp('STEP 4.1: configuring IT measurement class...');
disp(' ');

% ******************************

noise_level = 'class_0.1';  % 'class_0.1' or 'class_0.5'- default value is 'class_0.1'.
Bad_Data = false; % true or false - default value is 'false'

% ******************************

disp('STEP 4.2a: configuring LWLS estimator ...');
disp(' ');

% Select PMU locations
idx_LSE = 1:n_nodes;
%idx_LSE = [1, 3, 5, 7, 9, 10, 11, 12];

% Each PMU measures voltage and current
idx_LSE_V = idx_LSE;
idx_LSE_I = idx_LSE;

% Standard deviations of PMU noise w.r.t. real/imaginary part
switch noise_level
    case 'class_0.1'
        sd_PMU_V = 1e-3/3;
        sd_PMU_I = 1e-3/3;
   
    case 'class_0.5'
        sd_PMU_V = 5e-3/3;
        sd_PMU_I = 5e-3/3;
    otherwise 
        warning('Unexpected PMU class.')
end

n_LSE_meas = 2*(length(idx_LSE_V)+length(idx_LSE_I));

disp(['The LWLS estimator has ' int2str(n_LSE_meas) ' measurements.']);
disp(' ');

disp('STEP 4.2b: configuring NWLS estimator ...');
disp(' ');

% Select PMU/RTU locations
idx_NSE = idx_LSE;
%idx_NSE = [1, 3, 5, 7, 9, 10, 11, 12];

% Each PMU measures voltage magnitude and phase
% Each RTU measures active and reactive power
idx_NSE_V = idx_NSE;
idx_NSE_delta = idx_NSE;
idx_NSE_PQ = idx_NSE;

switch noise_level
    case 'class_0.1'
        % Standard deviation of PMU noise w.r.t. phase angle
        sd_PMU_delta = 15e-3/3;
        % Standard deviation of RTU noise
        sd_RTU_PQ = 1e-2/3;
   
    case 'class_0.5'
        % Standard deviation of PMU noise w.r.t. phase angle
        sd_PMU_delta = 90e-3/3;
        % Standard deviation of RTU noise
        sd_RTU_PQ = 5e-2/3;
    otherwise 
        warning('Unexpected measurement class.')
end

n_NSE_meas = length(idx_NSE_V)+length(idx_NSE_delta)+2*length(idx_NSE_PQ);

disp(['The NWLS estimator has ' int2str(n_NSE_meas) ' measurements.']);
disp(' ');

%% Step 5.1: Solve load flow problem

disp('STEP 5.1: solving LF problem ...');
disp(' ');

% Node types
idx_LF_PQ = 2:n_system; % everything except the slack node
idx_LF_PV = []; % none

% Parameters for Newton-Raphson method
max_iter = 100;
tol = 1e-10;

% Solve load flow for true nodal voltage phasors

V_LF = zeros(n_system,n_timesteps); % true nodal voltage phasors

for t=1:n_timesteps
    S_star = S_net(:,t);
    
    if(t<=1)
        V_0 = ones(n_system,1);
    else
        V_0 = V_LF(:,t-1);
    end
    
    [V_LF(:,t),J,n] = do_LF(Y_LF,S_star,V_0,idx_LF_slack,idx_LF_PQ,idx_LF_PV,tol,max_iter);
end

V_LF(idx_LF_slack,:) = []; % remove slack node
I_LF = Y_SE * V_LF;
S_LF = V_LF .* conj(I_LF);

%% Step 5.2: Emulate measurement units

disp('STEP 5.2a: emulating PMUs ...');
disp(' ');

V_PMU = add_noise_rectangular(V_LF(idx_LSE_V,:),sd_PMU_V,sd_PMU_V,Bad_Data);
I_PMU = add_noise_rectangular(I_LF(idx_LSE_I,:),sd_PMU_I,sd_PMU_I,0);

disp('STEP 5.2b: emulating RTUs ...');
disp(' ');

V_RTU = add_noise_polar(V_LF(idx_NSE_V,:),sd_PMU_V,sd_PMU_delta);
S_RTU = add_noise_rectangular(S_LF(idx_NSE_PQ,:),sd_RTU_PQ,sd_RTU_PQ,0);

%% Step 5.3: Run state estimation

disp('STEP 5.3: Running SE ...');
disp(' ');

V_LWLS = zeros(n_nodes,n_timesteps);
V_NWLS = zeros(n_nodes,n_timesteps);

t_LWLS = zeros(1,n_timesteps);
t_NWLS = zeros(1,n_timesteps);

V_0 = ones(n_nodes,1);
detected_bad_data = zeros(1,n_timesteps);

for t=1:n_timesteps %hk is the time-step index
    % LWLS
    t_start = tic;
    [V_LWLS(:,t),detected_bad_data(t)] = SE_LWLS(Y_SE,V_PMU(:,t),I_PMU(:,t),idx_LSE_V,idx_LSE_I,sd_PMU_V,sd_PMU_I,Bad_Data);
    t_LWLS(t) = toc(t_start);
    
    % NWLS
    t_start = tic;
    V_NWLS(:,t) = SE_NWLS(Y_SE,V_RTU(:,t),S_RTU(:,t),idx_NSE_V,idx_NSE_PQ,sd_PMU_V,sd_PMU_delta,sd_RTU_PQ,V_0,max_iter,tol);
    t_NWLS(t) = toc(t_start);
end

%%  Step 6: Post-processing and visualization

disp('STEP 6: analysing and plotting ...');
disp(' ');

folder = pwd();

%% Estimator Performance

show_PMU = 1;
show_RTU = 1;

h_timeseries = plot_timeseries(1,show_PMU,show_RTU,idx_LSE,idx_NSE,V_LF,V_PMU,V_RTU,V_LWLS,V_NWLS);
saveas(h_timeseries,[folder filesep() 'Results_Timeseries'],'epsc');

%% Estimation Error

h_error = plot_error(2,V_LF,V_LWLS,V_NWLS);
saveas(h_error,[folder filesep() 'Results_Estimation_Error'],'epsc');

%% Execution Time

disp(['LWLS: mean execution time = ' num2str(mean(t_LWLS)*1000,'%4.3f') 'ms.']);
disp(['NWLS: mean execution time = ' num2str(mean(t_NWLS)*1000,'%4.3f') 'ms.']);
disp(' ');

%% Bad data detection
[bd_index] = find(detected_bad_data);
corrupted_measurements = detected_bad_data(bd_index);
if corrupted_measurements
    disp(['The bad data is measurement: ' num2str(median(corrupted_measurements)) '.']);
else
    disp(['There is no bad data present.']);
end
